/* Copyright (c) 2016-2018 VMware, Inc. All rights reserved. */
package com.vmware.automatic.plugin.registration.commands;

import com.vmware.automatic.plugin.registration.commands.utils.PrintHelper;
import com.vmware.vim25.Description;
import com.vmware.vim25.Extension;
import com.vmware.vim25.ExtensionClientInfo;
import com.vmware.vim25.ExtensionResourceInfo;
import com.vmware.vim25.ExtensionServerInfo;
import com.vmware.vim25.KeyValue;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;

/**
 * Parsing and prepare command line arguments to operation register plugin.
 * ./extension-registration -action registerPlugin
 * -url <https://IP_OF_THE_VCENTER/sdk/>                - required argument, vCenter server IP
 * -username <USERNAME>                                 - required argument, vCenter server username
 * -password <PASSWORD>                                 - required argument, vCenter server password
 * -pluginUrl <https://DOWNLOAD_LOCATION/MY_PLUGIN.zip> - required argument, location of plugin zip
 * -key <MY_PLUGIN_KEY>                                 - required argument, plugin key
 * -version <PLUGIN_VERSION>                            - required argument, plugin version
 * -serverThumbprint <SERVER_THUMB_PRINT>               - required argument when pluginUrl is https, Thumbprint of plugin download server
 * -name <PLUGIN_NAME>                                  - optional argument, plugin name
 * -summary <PLUGIN_SUMMARY>                            - optional argument, plugin summary
 * -company <PLUGIN_COMPANY>                            - optional argument, plugin developer
 * -showInSolutionManager                               - optional argument, if specified the plugin will be shown in Administration panel, Solutions plug-in list
 *
 */
public class RegisterPluginCmd extends PluginCmd {
   /**
    * Extension plugin type
    */
   private static final String DEFAULT_PLUGIN_TYPE = "vsphere-client-serenity";
   /**
    * Extension server protocol must match to plugin _url protocol
    */
   private static final String HTTPS_PROTOCOL = "HTTPS";
   /**
    * Extension server adminEmail
    */
   private static final String ADMIN_EMAIL = "noreply@vmware.com";

   private String _pluginUrl;
   private String _name;
   private String _summary;
   private String _version;
   private String _company;
   private boolean _showInSolutionManager;
   private String _serverThumbprint;

   public RegisterPluginCmd() {
      _action = "-action registerPlugin";
      _errorMessage = "Register plugin to the VC";
   }

   public Options getPluginOpts() {
      Options options = super.getPluginOpts();
      options.addOption(Option.builder("pu").longOpt("pluginUrl").hasArg().argName("plugin url")
            .desc("Url from where the plugin will be downloaded").required().build());
      options.addOption(Option.builder("v").longOpt("version").hasArg().argName("version").desc("Plugin version")
            .required().build());
      options.addOption(Option.builder("n").longOpt("name").hasArg().argName("name").desc("Plugin name")
            .required(false).build());
      options.addOption(Option.builder("s").longOpt("summary").hasArg().argName("summary").desc("Plugin summary")
            .required(false).build());
      options.addOption(Option.builder("c").longOpt("company").hasArg().argName("company").desc("Plugin company")
            .required(false).build());
      options.addOption(Option.builder("st").longOpt("serverThumbprint").hasArg().argName("server thumbprint")
            .desc("Thumbprint of the server from which the plugin will be downloaded.").required(false).build());
      options.addOption(Option.builder("show").longOpt("showInSolutionManager")
            .desc("If specified the plugin will be shown in Administration panel, Solutions plug-in list")
            .required(false).build());
      return options;
   }

   protected void populateFields(String[] commandLineArgs) {
      super.populateFields(commandLineArgs);
      if (_commandLine != null) {
         _pluginUrl = _commandLine.getOptionValue("pu");
         String nameCmd = _commandLine.getOptionValue("n");
         if (nameCmd != null) {
            _name = nameCmd;
         }
         String summary = _commandLine.getOptionValue("s");
         if (summary != null) {
            _summary = summary;
         }
         String version = _commandLine.getOptionValue("v");
         if (version != null) {
            _version = version;
         }
         String company = _commandLine.getOptionValue("c");
         if (company != null) {
            _company = company;
         }
         _showInSolutionManager = _commandLine.hasOption("show");
         String serverThumbprint = _commandLine.getOptionValue("st");
         if (serverThumbprint != null) {
            _serverThumbprint = serverThumbprint;
         }
      }
   }


   /**
    * Registers a single extension based on the set member fields.
    */
   protected void doExecute() {
      Extension extension = new Extension();

      Description description = new Description();
      description.setLabel((_name != null) ? _name : "");
      description.setSummary((_summary != null) ? _summary : "");

      extension.setKey(_key);
      extension.setVersion((_version != null) ? _version : "");
      extension.setCompany(_company);
      extension.setDescription(description);

      ExtensionClientInfo extClientInfo = new ExtensionClientInfo();
      extClientInfo.setVersion((_version != null) ? _version : "");
      extClientInfo.setCompany((_company != null) ? _company : "");
      extClientInfo.setDescription(description);
      extClientInfo.setType(DEFAULT_PLUGIN_TYPE);
      extClientInfo.setUrl(_pluginUrl);
      extension.getClient().add(extClientInfo);

      ExtensionResourceInfo extResourceInfo_en = new ExtensionResourceInfo();
      extResourceInfo_en.setLocale("en_US");
      extResourceInfo_en.setModule("name");
      KeyValue kv1 = new KeyValue();
      kv1.setKey("name");
      kv1.setValue((_name != null) ? _name : "");
      extResourceInfo_en.getData().add(kv1);
      extension.getResourceList().add(extResourceInfo_en);

      if (_pluginUrl.startsWith(HTTPS_PROTOCOL.toLowerCase())) {
         // HTTPS requests require server info
         if (_serverThumbprint == null) {
            PrintHelper.printHelp(getPluginOpts(), "Missing required option: -serverThumbprint required argument when -pluginUrl is https");
            return;
         }
         ExtensionServerInfo extServerInfo = new ExtensionServerInfo();
         extServerInfo.getAdminEmail().add(ADMIN_EMAIL);
         extServerInfo.setCompany((_company != null) ? _company : "");
         extServerInfo.setDescription(description);
         extServerInfo.setType(HTTPS_PROTOCOL);
         extServerInfo.setServerThumbprint(_serverThumbprint);
         extServerInfo.setUrl(_pluginUrl);
         extension.getServer().add(extServerInfo);
      } else {
         System.out.println("INFO: Not using https for your plugin URL is OK for testing but not recommended for production." +
               "\nUsers will have to include the flag allowHttp=true in their vSphere Client webclient.properties otherwise the http URL will be ignored");
      }
      extension.setShownInSolutionManager(_showInSolutionManager);
      GregorianCalendar cal = new GregorianCalendar(TimeZone.getTimeZone("GMT"));
      try {
         DatatypeFactory dtFactory = DatatypeFactory.newInstance();
         XMLGregorianCalendar xmlCalendar = dtFactory.newXMLGregorianCalendar(cal);
         extension.setLastHeartbeatTime(xmlCalendar);
         _vimPort.registerExtension(_extensionManager, extension);
         System.out.println("Plugin: " + _key + " has been successfully registered in vCenter <" + _url + ">.");
      } catch (RuntimeFaultFaultMsg e) {
         System.out.println("Exception while registering plugin " + _key);
         e.printStackTrace();
      } catch (DatatypeConfigurationException e) {
         System.out.println("Exception while registering plugin" + _key);
         e.printStackTrace();
      }
   }
}
